/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Notifications generated by protocol(s)
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   MPC5xx Framework
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file ProtocolNotify.h
    \brief Certain protocols have particular states that are of interest to an application. Notification events
           are provides to allow an application to execute code when these states are detected.
           
 For example, a Diagnostic Function Manager (DFM) would want to know of an ERI protocol exception because this
 exception generally means that the communications cable has been disconnected or failed. The manager would
 need to be aware of such a condition so that it could cancel any active functions that it was managing.

 See \ref E_ProtocolNotifyEvents for a list of supported notification events\n
 See RegisterProtocolNotification() for a description of how a notification can be created. */

#ifndef __PROTOCOLNOTIFY_H
#define __PROTOCOLNOTIFY_H

/*---- INCLUDE FILES ----------------------------------------------------------------------------------------*/
#include <typedefn.h>

/*---- DEFINES ----------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration identifes the supported protocol events.

    The values are organised so that they can be used in a mask. This allows a notification callback to be
    defined that will execute off multiple events.
    \code
    E_ProtocolNotifyEvents eDesiredEventMask = PROTOCOLNOTIFYEV_KEYCONNECTED | PROTOCOLNOTIFYEV_KEYEXCEPTION;
    \endcode
*/
typedef enum
{
/*! This event is issued when a connection to the KEY protocol is successfully established. The supporting data
    associated with this event is undefined */
    PROTOCOLNOTIFYEV_KEYCONNECTED       = 0x0001,
/*! This event is issued when a successful security exchange has occurred. */
    PROTOCOLNOTIFYEV_KEYACCESS_UNLOCKED = 0x0002,
/*! This event is issued when the KEY protocol first allows a security exchange to occur. Attempts to establish
    a secure connection with KEY prior to the receuipt of this event will fail. */
    PROTOCOLNOTIFYEV_KEYACCESS_LOCKED   = 0x0004,
/*! The KEY protocol has had an exception. The supporting data describes an E_Exceptions */
    PROTOCOLNOTIFYEV_KEYEXCEPTION       = 0x0008,
/*! The ERI protocol has had an exception. The supporting data describes an E_Exceptions. The underlying
    protocol will be forced to revert back to KEY. */
    PROTOCOLNOTIFYEV_ERIEXCEPTION       = 0x0010,
/*! The MOTPROG protocol has had an exception. The supporting data describes an E_Exceptions. The underlying
    protocol will be forced to revert back to KEY. */
    PROTOCOLNOTIFYEV_MOTPROGEXCEPTION   = 0x0020,
/*! The KEY protocol has entered the blocking state where further attempts to negotiate a secure connection
    will automatically fail */
    PROTOCOLNOTIFYEV_KEYBLOCKED_LOCKED  = 0x0040,
/*! The CANKING protocol has had an exception. The supporting data describes an E_Exceptions. The underlying
    protocol will be forced to revert back to KEY. */
    PROTOCOLNOTIFYEV_CANKINGEXCEPTION   = 0x0080,
/*! A KEY-KEY exchange has occurred. A remote device sends 0x55 and the module replies with 0xAA. If is a ping,
    and is usually done to verify the validity of the link. */
    PROTOCOLNOTIFYEV_KEYKEY             = 0x0200,

/*! A WOODWARD_KEY exchange has occurred. A remote device sends [0x01, 0x6X, 0x0X, 0xXX, 0x03, 0xXX, 0xXX] and 
    the module is required to jump to Woodward Boot Code. */
    PROTOCOLNOTIFYEV_WOODWARDKEY        = 0x0400,

    PROTOCOLNOTIFYEV_ALL_EVENTS         = 0xFFFF,
} E_ProtocolNotifyEvents;

#define RegisterProtocolNotification(in_pfNotify, in_eEventMask, in_uAppDataToSendWithNotify) \
     RegisterProtocolNotificationInstance(in_pfNotify, 0, in_eEventMask, in_uAppDataToSendWithNotify)

#define UnRegisterProtocolNotification(in_pfNotify) \
     UnRegisterProtocolNotificationInstance(in_pfNotify, 0)

/*---- TYPEDEF ----------------------------------------------------------------------------------------------*/
/*! \brief Notification function type

 A function of this type is supplied as one of the parameters of the RegisterProtocolNotificationInstance().
 It is executed when the specified event(s) is detected. \p in_u1City describes the City of the connection that
 detected the event. Protocols executing over CAN can operate concurrently. The City describes their address.
 \ref E_ProtocolNotifyEvents describes the event that was detected. \p in_u2SupportData describes data specific
 to the event that has been detected. For example, an exception notification event like PROTOCOLNOTIFYEV_KEYEXCEPTION
 would hold the actual exception that occurred. \p in_uAppSuppliedData is the application data that was supplied
 when the notification event was registered. */
typedef void (*ProtocolNotificationFuncPtr)(uint1 in_u1City, E_ProtocolNotifyEvents, uint2 in_u2SupportData, NativePtrSizedInt_U in_uAppSuppliedData);

#pragma pack()
/*---- EXTERNALS --------------------------------------------------------------------------------------------*/

/*---- PROTOTYPES -------------------------------------------------------------------------------------------*/

NativeBool RegisterProtocolNotificationInstance(ProtocolNotificationFuncPtr, uint2 in_u2Instance, E_ProtocolNotifyEvents, NativePtrSizedInt_U in_uAppDataToSendWithNotify);
NativeBool UnRegisterProtocolNotificationInstance(ProtocolNotificationFuncPtr, uint2 in_u2Instance);

#endif /* __PROTOCOLNOTIFY_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/
